#include <stdio.h>

#define VERSION "V2.21"
#define HEADINGS 72

#define MAXVERSIONS 13
#define MAXATTACH 8
#define MAXSUBDATA 6
#define DESCSIZE 0x0448e0
#define LASTITEM 0x015e

struct offsets
{
        char    *name;
        char    *itemname;
        char    *bobbyname;
        long    checkdata[4];
        long    checksize;
        long    weaponattach;
        long    otherattach;
        long    ammoattach;
        long    exclusions;
        long    combo2way;
        long    combo3way;
        long    convertweapons;
        long    convertammo;
        long    endammo;
        long    calibre1;
        long    calibre2;
        long    sounds;
};

struct offsets offsets[] = {
       "V1.07",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4C010600,0x08C13339,0x00000000,
       2961920,
       0x2383c0,
       0x238698,
       0x238708,
       0x238738,
       0x238798,
       0x238940,
       0x238968,
       0x2389b8,
       0x2389d8,
       0x23af40,
       0x23b1e8,
       0x263180,

       "V1.06 Wildfire",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4c010600,0xa1663038,0x00000000,
       2961965,
       0x2383a8,
       0x238680,
       0x2386f0,
       0x238720,
       0x238780,
       0x238928,
       0x238950,
       0x2389a0,
       0x2389c0,
       0x23af28,
       0x23b1d0,
       0x263168,

       "V1.06",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4c010600,0x31393131,0x00000000,
       2827776,
       0x2383a8,
       0x238680,
       0x2386f0,
       0x238720,
       0x238780,
       0x238928,
       0x238950,
       0x2389a0,
       0x2389c0,
       0x23af28,
       0x23b1d0,
       0x263168,

       "V1.05 Wildfire",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4c010600,0xff83cd37,0x00000000,
       2826240,
       0x236320,
       0x2365f8,
       0x236668,
       0x236698,
       0x2366f8,
       0x2368a0,
       0x2368c8,
       0x236918,
       0x236938,
       0x239038,
       0x2392e0,
       0x261278,

       "V1.05",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4c010600,0x31393131,0x00000000,
       2826240,
       0x236320,
       0x2365f8,
       0x236668,
       0x236698,
       0x2366f8,
       0x2368a0,
       0x2368c8,
       0x236918,
       0x236938,
       0x239038,
       0x2392e0,
       0x261278,

       "V1.05r",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4c010600,0x31393131,0x00000000,
       2818048,
       0x236320,
       0x2365f8,
       0x236668,
       0x236698,
       0x2366f8,
       0x2368a0,
       0x2368c8,
       0x236918,
       0x236938,
       0x239038,
       0x2392e0,
       0x261278,

       "V1.02 Dutch",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x1148af7c,0x5529c12f,0x5529c12f,0x5529c12f,
       2703360,
       0x21bd24,
       0x21bffc,
       0x21c06c,
       0x21c098,
       0x21c0f8,
       0x21c298,
       0x21c2c0,
       0x21c30c,
       0x21c328,
       0x21e9cc,
       0x21ec74,
       0x247874,

       "V1.02 German",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4C010600,0xFA496137,0x00000000,
       2919936,
       0x22dec8,
       0x22e1a0,
       0x22e210,
       0x22e240,
       0x22e2a0,
       0x22e440,
       0x22e468,
       0x22e4b8,
       0x22e4d8,
       0x230bd8,
       0x230e80,
       0x259e90,

       "V1.02b German",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x50450000,0x4C010600,0xDA925537,0x00000000,
       2924544,
       0x22ee28,
       0x22f100,
       0x22f170,
       0x22f1a0,
       0x22f200,
       0x22f3a0,
       0x22f3c8,
       0x22f418,
       0x22f438,
       0x2319a0,
       0x231c48,
       0x25aca8,

       "V1.05 German",
       "data/binarydata/itemdesc.edt",
       "data/binarydata/braydesc.edt",
       0x708a2be3,0x34EB45B0,0x34EB45B0,0x34EB45B0,
       2719775,
       0x21df5c,
       0x21e234,
       0x21e2a4,
       0x21e2d0,
       0x21e330,
       0x21e4d8,
       0x21e500,
       0x21e54c,
       0x21e568,
       0x220a74,
       0x220d1c,
       0x249be4,

       "UB V1.00",
       "data/binarydata/itmdsc25.edt",
       "",
       0xffdeacc2,0xbbbfc291,0xbbbfc291,0xbbbfc291,
       2940962,
       0x23c13c,
       0x23c4c8,
       0x23c53c,
       0x23c570,
       0x23c5e0,
       0x23c788,
       0x23c7b8,
       0x23c804,
       0x23c820,
       0x23eb78,
       0x23ee98,
       0x262914,

       "UB V1.01",
       "data/binarydata/itmdsc25.edt",
       "",
       0x7aa7ac3f,0x3ec6c26c,0x3ec6c26c,0x3ec6c26c,
       2940962,
       0x23c134,
       0x23c4c0,
       0x23c534,
       0x23c568,
       0x23c5d8,
       0x23c780,
       0x23c7b0,
       0x23c7fc,
       0x23c818,
       0x23eb5c,
       0x23ee7c,
       0x2628f8,

       "UB V1.01 Russian",
       "data/binarydata/itmdsc25.edt",
       "",
       0xAF1A048D,0xEB7B6ADE,0xEB7B6ADE,0xEB7B6ADE,
       2936866,
       0x23c134,
       0x23c4c0,
       0x23c534,
       0x23c568,
       0x23c5d8,
       0x23c780,
       0x23c7b0,
       0x23c7fc,
       0x23c818,
       0x23eb5c,
       0x23ee7c,
       0x261738,

};

struct attribtype
{
        long    offset;
        long    start;
        long    end;
        long    size;
        char    *name;
};

struct attribtype attribtype[] = {
       0x2389d8,0x0000,0x0045,0x001a,"Weapons",
       0x236798,0x0000,0x015e,0x0014,"All Items",
       0x2390f8,0x0047,0x0076,0x0003,"Ammo Clips",
       0x239188,0x00a1,0x00c6,0x0003,"Armour",
       0x2391f8,0x0083,0x00a0,0x0007,"Explosives",
       0x279e18,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x2389c0,0x0000,0x0045,0x001a,"Weapons",
       0x236780,0x0000,0x015e,0x0014,"All Items",
       0x2390e0,0x0047,0x0076,0x0003,"Ammo Clips",
       0x239170,0x00a1,0x00c6,0x0003,"Armour",
       0x2391e0,0x0083,0x00a0,0x0007,"Explosives",
       0x279e00,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x2389c0,0x0000,0x0045,0x001a,"Weapons",
       0x236780,0x0000,0x015e,0x0014,"All Items",
       0x2390e0,0x0047,0x0076,0x0003,"Ammo Clips",
       0x239170,0x00a1,0x00c6,0x0003,"Armour",
       0x2391e0,0x0083,0x00a0,0x0007,"Explosives",
       0x279e00,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x236938,0x0000,0x0045,0x001a,"Weapons",
       0x2346f8,0x0000,0x015e,0x0014,"All Items",
       0x237058,0x0047,0x0076,0x0003,"Ammo Clips",
       0x2370e8,0x00a1,0x00c6,0x0003,"Armour",
       0x237158,0x0083,0x00a0,0x0007,"Explosives",
       0x277e50,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x236938,0x0000,0x0045,0x001a,"Weapons",
       0x2346f8,0x0000,0x015e,0x0014,"All Items",
       0x237058,0x0047,0x0076,0x0003,"Ammo Clips",
       0x2370e8,0x00a1,0x00c6,0x0003,"Armour",
       0x237158,0x0083,0x00a0,0x0007,"Explosives",
       0x277e50,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x236938,0x0000,0x0045,0x001a,"Weapons",
       0x2346f8,0x0000,0x015e,0x0014,"All Items",
       0x237058,0x0047,0x0076,0x0003,"Ammo Clips",
       0x2370e8,0x00a1,0x00c6,0x0003,"Armour",
       0x237158,0x0083,0x00a0,0x0007,"Explosives",
       0x277e50,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x21c328,0x0000,0x0045,0x001a,"Weapons",
       0x21a100,0x0000,0x015e,0x0014,"All Items",
       0x21ca44,0x0047,0x0076,0x0003,"Ammo Clips",
       0x21cad0,0x00a1,0x00c6,0x0003,"Armour",
       0x21cb41,0x0083,0x00a0,0x0007,"Explosives",
       0x25e3b0,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x22e4d8,0x0000,0x0045,0x001a,"Weapons",
       0x22c2a0,0x0000,0x015e,0x0014,"All Items",
       0x22ebf8,0x0047,0x0076,0x0003,"Ammo Clips",
       0x22ec88,0x00a1,0x00c6,0x0003,"Armour",
       0x22ecf8,0x0083,0x00a0,0x0007,"Explosives",
       0x270b38,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x22f438,0x0000,0x0045,0x001a,"Weapons",
       0x22d200,0x0000,0x015e,0x0014,"All Items",
       0x22fb58,0x0047,0x0076,0x0003,"Ammo Clips",
       0x22fbe8,0x00a1,0x00c6,0x0003,"Armour",
       0x22fc58,0x0083,0x00a0,0x0007,"Explosives",
       0x271a10,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x21e568,0x0000,0x0045,0x001a,"Weapons",
       0x21c338,0x0000,0x015e,0x0014,"All Items",
       0x21ec84,0x0047,0x0076,0x0003,"Ammo Clips",
       0x21ed10,0x00a1,0x00c6,0x0003,"Armour",
       0x21ed81,0x0083,0x00a0,0x0007,"Explosives",
       0x2608b8,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x23c820,0x0000,0x0046,0x001a,"Weapons",
       0x23a518,0x0000,0x015e,0x0014,"All Items",
       0x23cf58,0x0047,0x0078,0x0003,"Ammo Clips",
       0x23cff4,0x00a1,0x00c6,0x0003,"Armour",
       0x23d069,0x0083,0x00a0,0x0007,"Explosives",
       0x28829c,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x23c818,0x0000,0x0046,0x001a,"Weapons",
       0x23a510,0x0000,0x015e,0x0014,"All Items",
       0x23cf50,0x0047,0x0078,0x0003,"Ammo Clips",
       0x23cfec,0x00a1,0x00c6,0x0003,"Armour",
       0x23d061,0x0083,0x00a0,0x0007,"Explosives",
       0x288284,0x0000,0x015e,0x0002,"Bobby Ray's",

       0x23c818,0x0000,0x0046,0x001a,"Weapons",
       0x23a510,0x0000,0x015e,0x0014,"All Items",
       0x23cf50,0x0047,0x0078,0x0003,"Ammo Clips",
       0x23cfec,0x00a1,0x00c6,0x0003,"Armour",
       0x23d061,0x0083,0x00a0,0x0007,"Explosives",
       0x286c84,0x0000,0x015e,0x0002,"Bobby Ray's",

};


struct attriblist
{
        short   offset;
        char    size;
        char    *name;
};

struct attriblist attriblist[0][30] = {
        0x0000,1,"Class1",
        0x0001,1,"Class2",
	0x0002,1,"Ammo Type",
	0x0003,1,"Draw Cost",
	0x0004,1,"Fire Rate",
	0x0005,1,"Burst Size",
	0x0006,1,"Burst Pen",
	0x0007,1,"Unknown",
	0x0008,1,"Damage",
	0x0009,1,"Prestige",
	0x000a,1,"Unknown",
	0x000b,1,"Clip Size",
	0x000c,2,"Range",
	0x000e,1,"Unknown",
	0x000f,1,"Unknown",
	0x0010,1,"Loudness",
	0x0011,1,"Unknown",
	0x0012,1,"Fire Sound",
	0x0013,1,"Unknown",
	0x0014,1,"Burst Sound (Unused)",
	0x0015,1,"Unknown",
	0x0016,1,"Class3",
	0x0017,1,"Unknown",
	0x0018,1,"Class4",
	0x0019,1,"Unknown",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",

        0x0000,1,"Attack Type",
        0x0001,1,"Usage",
	0x0002,1,"Unknown",
	0x0003,1,"Unknown",
	0x0004,1,"Special Offset",
	0x0005,1,"Cursor Type",
	0x0006,1,"Unknown",
	0x0007,1,"Graphic1",
	0x0008,1,"Graphic2",
	0x0009,1,"Weight",
	0x000a,1,"Size",
	0x000b,1,"Unknown",
	0x000c,2,"Price",
	0x000e,1,"Progress",
	0x000f,-1,"Durability",
	0x0010,-1,"Repair",
	0x0011,1,"Unknown",
	0x0012,1,"Hands",
	0x0013,1,"Availability",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",

        0x0000,1,"Clip Type",
        0x0001,1,"Clip Size",
        0x0002,1,"Clip Kind",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",

        0x0000,1,"Armour Type",
        0x0001,1,"Armour Strength",
        0x0002,1,"Armour Durabilty",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",

        0x0000,1,"Exp Type",
        0x0001,1,"Exp Damage",
        0x0002,1,"Exp Breath",
        0x0003,1,"Exp Radius",
        0x0004,1,"Unknown (Sound?)",
        0x0005,1,"Stability",
        0x0006,1,"Animation",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",

        0x0000,1,"Bobby Ray's Stock",
        0x0001,1,"BR's Used Stock",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
        0x0000,0,"",
};


struct weaplist
{
	short	offset;
	char	*name;
};


struct  weaplist attachtypes[]={
	0x00CF,"Silencer",
	0x00F1,"Laser Scope",
	0x00D0,"Sniper Scope",
	0x00D1,"Bipod",
	0x0032,"Talon",
	0x00F4,"Duckbill",
	0x0133,"Rod & Spring",
	0x0136,"Barrel Extender",
};


unsigned long   attachments[MAXATTACH][1024];

unsigned char	*buffer;
unsigned char	*descbuffer;
unsigned char	*braybuffer;

char    dbuffer[1024];
long    ver;

int findattach(short weapon, short attach)
{
        extern  unsigned char	*buffer;
	unsigned long	c;

        for(c = offsets[ver].weaponattach ; c < offsets[ver].otherattach ; c += 4){
              if(*(short *)(buffer + c) == attach) if(*(short *)(buffer + c + 2) == weapon) return(1);
        }

        return(0);
}

int addattach(unsigned long weapon, unsigned long attach)
{
        extern  unsigned char	*buffer;
	unsigned long	c,c1;

        for(c = 0 ; ; c ++){
              if(attachtypes[c].offset == attach) break;
        }

	for(c1 = 0 ; ; c1 ++){
              if(!attachments[c][c1]) break;
        }

        attachments[c][c1] = (weapon << 16) + attach;

        return(0);
}


main(int argc,char *argv[])
{
        extern  unsigned char	*buffer;
        long	buffpoint;
	long	buffsize;
	FILE	*exefile;
	FILE	*descfile;
	FILE	*brayfile;
	FILE	*desc2file;
	FILE	*textfile;
	long	c,c1,c2;
        long	temp,temp2,temp3;
        char     ctemp;
        char     *ptemp;
        char     stemp[256];

	printf("\nJagged Alliance Weapon Editor %s\n\n",VERSION);

        for(c = 0 ; c < MAXATTACH ; c++){
                for(c1 = 0 ; c1 < (offsets[ver].otherattach - offsets[ver].weaponattach) / 4 ; c1++){
                       attachments[c][c1] = 0;
                }
        }


        if(argc < 4){
help:
		printf("\nUsage: ja2wedit <executable> <textfile> <[make]|[patch]>\n\n");
		printf("Use the MAKE option to dump stats to a tab delimited text file.\n");
		printf("To apply the changes from the textfile, use the PATCH option.\n");
		return(0);
	}


        if(!(exefile = fopen(argv[1],"rb"))){
		printf("Can't open %s!\n",argv[1]);
		goto help;
	}
	fseek(exefile,0,SEEK_END);
	buffsize = ftell(exefile);
        for(ver = 0 ; ver < MAXVERSIONS ; ver++){
                if(offsets[ver].checksize == buffsize){
                       fseek(exefile,0x80,SEEK_SET);
                       for (c = 0 ; c < 4 ; c++){
                           fread(&temp2,4,1,exefile);
                           temp = ((temp2 >> 24) & 0x000000ff) + ((temp2 & 0x00ff0000) >> 8) + ((temp2 & 0x0000ff00) << 8) + (temp2 << 24);
                           if(temp != offsets[ver].checkdata[c]) goto notversion;
                       }
                       printf("%s version of executable detected\n",offsets[ver].name);
                       break;
                }
        notversion:
        }
	fseek(exefile,0,SEEK_SET);
        if(ver >= MAXVERSIONS){
        	printf("Unrecognized version of executable.\n");
		fclose(exefile);
		return(1);
	}
	buffer = (char *) malloc(buffsize);
	if(!buffer){
		printf("Unable to allocate %d bytes.\n",buffsize);
		fclose(exefile);
		return(1);
	}

        printf("\nReading binary data...");

	c = fread(buffer,1,buffsize,exefile);
	if (c != buffsize){
		printf("Error while reading executable (Read %d of %d bytes).\n",c,buffsize);
		fclose(exefile);
		return(1);
	}
	fclose(exefile);

	if(!stricmp(argv[3],"make")){
                printf("\nReading item descriptions...");

                if(!(descfile = fopen(offsets[ver].itemname,"rb"))){
                        printf("Error reading item descriptions.\nYou must have the %s file installed.\n",offsets[ver].itemname);
                        return(1);
                }
                descbuffer = (char *) malloc(DESCSIZE);
	        if(!descbuffer){
                        printf("Unable to allocate %d bytes.\n",DESCSIZE);
		        fclose(descfile);
		        return(1);
                }
                c = fread(descbuffer,1,DESCSIZE,descfile);
	        if (c != DESCSIZE){
                        printf("Error while reading item descriptions (Read %d of %d bytes).\n",c,DESCSIZE);
		        fclose(descfile);
		        return(1);
                }
	        fclose(descfile);

                if(strlen(offsets[ver].bobbyname)){
                        if(!(brayfile = fopen(offsets[ver].bobbyname,"rb"))){
                               printf("Error reading Bobby Ray's item descriptions.\nYou must have the %s file installed.\n",offsets[ver].bobbyname);
                               return(1);
                        }
                        braybuffer = (char *) malloc(DESCSIZE);
	                if(!braybuffer){
                               printf("Unable to allocate %d bytes.\n",DESCSIZE);
		               fclose(brayfile);
		               return(1);
                        }
                        c = fread(braybuffer,1,DESCSIZE,brayfile);
	                if (c != DESCSIZE){
                               printf("Error while reading Bobby Ray's item descriptions (Read %d of %d bytes).\n",c,DESCSIZE);
		               fclose(brayfile);
		               return(1);
                        }
                        fclose(brayfile);
                }

                printf("\nDumping weapon stats...");
		if(!(textfile = fopen(argv[2],"w"))){
			printf("Unable to create %s!\n",argv[2]);
			fclose(textfile);
			return(1);
		}

		fprintf(textfile,"Offset\tShort Name\tLong Name\tDescription\tBobby Ray's Name\tBobby Ray's Description");

                for(c = 0 ; c < MAXSUBDATA ; c++){
                        for(c1 = 0 ; ; c1++){
                               if(!attriblist[c][c1].size) break;
                               fprintf(textfile,"\t%s",attriblist[c][c1].name);
                        }
                }
		for(c = 0 ; c < MAXATTACH ; c++){
			fprintf(textfile,"\t%s?",attachtypes[c].name);
		}

		for(c = 0 ; c <= LASTITEM ; c++){
			fprintf(textfile,"\nItem %d\t",c);
		        for(c1 = 0 ; ; c1 += 2){
                                ctemp = *(descbuffer + (c * 0x0320) + c1);
                                if(ctemp > 0x20) ctemp--;
                                if(ctemp) fputc(ctemp,textfile);
                                else break;
                        }
                        fprintf(textfile,"\t");
		        for(c1 = 0 ; ; c1 += 2){
                                ctemp = *(descbuffer + (c * 0x0320) + c1 + 0xa0);
                                if(ctemp > 0x20) ctemp--;
                                if(ctemp) fputc(ctemp,textfile);
                                else break;
                        }
                        fprintf(textfile,"\t");
		        for(c1 = 0 ; ; c1 += 2){
                                ctemp = *(descbuffer + (c * 0x0320) + c1 + 0x0140);
                                if(ctemp > 0x20) ctemp--;
                                if(ctemp) fputc(ctemp,textfile);
                                else break;
                        }
                        fprintf(textfile,"\t");
                        if(strlen(offsets[ver].bobbyname)){
                                for(c1 = 0 ; ; c1 += 2){
                                        ctemp = *(braybuffer + (c * 0x0320) + c1);
                                        if(ctemp > 0x20) ctemp--;
                                        if(ctemp) fputc(ctemp,textfile);
                                        else break;
                                }
                                fprintf(textfile,"\t");
		                for(c1 = 0 ; ; c1 += 2){
                                        ctemp = *(braybuffer + (c * 0x0320) + c1 + 0xa0);
                                        if(ctemp > 0x20) ctemp--;
                                        if(ctemp) fputc(ctemp,textfile);
                                        else break;
                                }
                        }
                        else{
                                fprintf(textfile,"n/a\tn/a");
                        }

                        for(c1 = 0 ; c1 < MAXSUBDATA ; c1++){
                                for(c2 = 0 ; ; c2++){
                                        if(!attriblist[c1][c2].size) break;
                                        if(c < attribtype[c1 + ver * MAXSUBDATA].start || c > attribtype[c1 + ver * MAXSUBDATA].end){
                                             fprintf(textfile,"\tn/a");
                                        }
				        else{
                                             ptemp = (buffer + attribtype[c1 + ver * MAXSUBDATA].offset + ((c - attribtype[c1 + ver * MAXSUBDATA].start) * attribtype[c1 + ver * MAXSUBDATA].size) + attriblist[c1][c2].offset);
                                             switch (attriblist[c1][c2].size){
                                                    case 1:
                                                         fprintf(textfile,"\t%d",*(unsigned char*) (ptemp));
                                                         break;
                                                    case -1:
                                                         fprintf(textfile,"\t%d",*(signed char*) (ptemp));
                                                         break;
                                                    case 2:
                                                         fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                                                         break;
                                                    case -2:
                                                         fprintf(textfile,"\t%d",*(signed short*) (ptemp));
                                                         break;
                                             }
                                        }
                                }
                        }
                        for(c1 = 0 ; c1 < MAXATTACH ; c1++){
                               if(c <= attribtype[ver * MAXSUBDATA].end){
                                        if(findattach(c,attachtypes[c1].offset)) fprintf(textfile,"\t1");
                                        else fprintf(textfile,"\t0");
                               }
                               else fprintf(textfile,"\tn/a");
                        }
                }

                for(c = 0, temp = offsets[ver].otherattach ; temp < offsets[ver].ammoattach; c++, temp += 4){
                        fprintf(textfile,"\nAttach %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                }

                for(c = 0, temp = offsets[ver].ammoattach ; temp < offsets[ver].exclusions; c++, temp += 4){
                        fprintf(textfile,"\nAmmoAttach %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                }

                for(c = 0, temp = offsets[ver].exclusions ; temp < offsets[ver].combo2way; c++, temp += 4){
                        fprintf(textfile,"\nExclude %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                }

                for(c = 0, temp = offsets[ver].combo2way ; temp < offsets[ver].combo3way; c++, temp += 8){
                        fprintf(textfile,"\nCombine2 %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+4));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+6));
                }

                for(c = 0, temp = offsets[ver].combo3way ; temp < offsets[ver].convertweapons; c++, temp += 8){
                        fprintf(textfile,"\nCombine3 %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+4));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+6));
                }

                for(c = 0, temp = offsets[ver].convertweapons ; temp < offsets[ver].convertammo; c++, temp += 4){
                        fprintf(textfile,"\nConvertWeapon %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                }

                for(c = 0, temp = offsets[ver].convertammo ; temp < offsets[ver].endammo; c++, temp += 4){
                        fprintf(textfile,"\nConvertAmmo %d",c);
                        ptemp = buffer + temp;
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp));
                        fprintf(textfile,"\t%d",*(unsigned short*) (ptemp+2));
                }

                for(c = 0 ; c < 17 ; c++){
                        fprintf(textfile,"\nAmmoType %d\t",c);
		        for(c1 = 0 ; ; c1 += 2){
                                ctemp = *(buffer + offsets[ver].calibre2 + (c * 0x28) + c1);
                                if(ctemp) fputc(ctemp,textfile);
                                else break;
                        }
                        fprintf(textfile,"\t");
		        for(c1 = 0 ; ; c1 += 2){
                                ctemp = *(buffer + offsets[ver].calibre1 + (c * 0x28) + c1);
                                if(ctemp) fputc(ctemp,textfile);
                                else break;
                        }
                }

                for(c = 0 ; c < 291 ; c++){
                        fprintf(textfile,"\nSound %d",c);
                        fprintf(textfile,"\t%s",buffer + offsets[ver].sounds + (c * 0xff));
                }

                printf("\nDone!\n");
                fclose(textfile);
		return(0);
	}

	if(stricmp(argv[3],"patch")){
		printf("\nInvalid option.  Use either MAKE or PATCH\n");
		goto help;
	}

        printf("\nReading item descriptions...");

        if(!(descfile = fopen(offsets[ver].itemname,"rb"))){
                printf("Error reading item descriptions.\nYou must have the %s file installed.\n",offsets[ver].itemname);
                return(1);
        }
        descbuffer = (char *) malloc(DESCSIZE);
	if(!descbuffer){
                printf("Unable to allocate %d bytes.\n",DESCSIZE);
	        fclose(descfile);
	        return(1);
        }
        c = fread(descbuffer,1,DESCSIZE,descfile);
	if (c != DESCSIZE){
                printf("Error while reading item descriptions (Read %d of %d bytes).\n",c,DESCSIZE);
                fclose(descfile);
	        return(1);
        }
        fclose(descfile);
        if(strlen(offsets[ver].bobbyname)){
                if(!(brayfile = fopen(offsets[ver].bobbyname,"rb"))){
                       printf("Error reading Bobby Ray's item descriptions.\nYou must have the %s file installed.\n",offsets[ver].bobbyname);
                       return(1);
                }
                braybuffer = (char *) malloc(DESCSIZE);
                if(!braybuffer){
                       printf("Unable to allocate %d bytes.\n",DESCSIZE);
	               fclose(brayfile);
	               return(1);
                }
                c = fread(braybuffer,1,DESCSIZE,brayfile);
                if (c != DESCSIZE){
                       printf("Error while reading Bobby Ray's item descriptions (Read %d of %d bytes).\n",c,DESCSIZE);
                       fclose(brayfile);
	               return(1);
                }
                fclose(brayfile);
        }

	if(!(textfile = fopen(argv[2],"r"))){
        	printf("Unable to read %s!\n",argv[2]);
		fclose(textfile);
		return(1);
        }

        printf("\nConverting text file...");

        temp = 0;
        for(;;){
                ctemp = fgetc(textfile);
                if(ctemp == '\t') temp++;
                if(ctemp == '\n') break;
                if(feof(textfile) || ferror(textfile)){
			printf("Error while parsing headings!\n");
			fclose(textfile);
			return(1);
                }
	}

        if(temp != HEADINGS){
		printf("Error while parsing!  (%d Headings)\n",temp);
		printf("If you are using a textfile from an older version, you\n");
                printf("use MAKE to create a new textfile.  See documentation.\n");
		fclose(textfile);
		return(1);
        }

	for(;;){
                for(c1 = 0 ; c1 < 255 ; c1++){
ignorejunk:
                       stemp[c1] = getc(textfile);
                       if(feof(textfile)) goto exit;
                       /* Ignore junk Tabs/Spaces/Whatever at the ends of lines */
                       if(!c1){
                               if(stemp[0] < 'a' || stemp[0] > 'z'){
                                      if(stemp[0] < 'A' || stemp[0] > 'Z') goto ignorejunk;
                               }
                       }
                       if(stemp[c1] == ' '){
                               stemp[c1]=0;
                               break;
                       }
                }

                if(!stricmp("AmmoType",stemp)){
                        fscanf(textfile,"%d",&temp);
                        ctemp = getc(textfile);
                        for(c1 = 0 ; ; c1 += 2){
                               ctemp = getc(textfile);
                               if(ctemp == 0x22) ctemp = getc(textfile);
                               if(ctemp == '\t') ctemp = 0;
                               if(ctemp == '\n') ctemp = 0;
                               if(feof(textfile)) ctemp = 0;
                               *(buffer + (temp * 0x28) + offsets[ver].calibre2 + c1) = ctemp;
                               if(!ctemp) break;
                        }
                        for(c1 = 0 ; ; c1 += 2){
                               ctemp = getc(textfile);
                               if(ctemp == 0x22) ctemp = getc(textfile);
                               if(ctemp == '\t') ctemp = 0;
                               if(ctemp == '\n') ctemp = 0;
                               if(feof(textfile)) ctemp = 0;
                               *(buffer + (temp * 0x28) + offsets[ver].calibre1 + c1) = ctemp;
                               if(!ctemp) break;
                        }
                        continue;
                }

                if(!stricmp("Sound",stemp)){
                        fscanf(textfile,"%d",&temp);
                        ctemp = getc(textfile);
                        for(c1 = 0 ; ; c1++){
                               skip:
                               ctemp = getc(textfile);
                               if(feof(textfile)) goto exit;
                               if(ctemp == '\t') goto skip;
                               if(ctemp == 0x22) goto skip; /* quotes */
                               if(ctemp == '\n') ctemp = 0;
                               *(buffer + offsets[ver].sounds + (temp * 0xff) + c1) = ctemp;
                               if(!ctemp) break;
                        }
                        continue;
                }

                if(!stricmp("Attach",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 2 ; c1++){
                               ptemp = buffer + offsets[ver].otherattach + (temp * 4) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(!stricmp("AmmoAttach",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 2 ; c1++){
                               ptemp = buffer + offsets[ver].ammoattach + (temp * 4) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(!stricmp("Exclude",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 2 ; c1++){
                               ptemp = buffer + offsets[ver].exclusions + (temp * 4) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(!stricmp("Combine2",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 4 ; c1++){
                               ptemp = buffer + offsets[ver].combo2way + (temp * 8) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(!stricmp("Combine3",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 4 ; c1++){
                               ptemp = buffer + offsets[ver].combo3way + (temp * 8) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(!stricmp("ConvertWeapon",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 2 ; c1++){
                               ptemp = buffer + offsets[ver].convertweapons + (temp * 4) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(!stricmp("ConvertAmmo",stemp)){
                        fscanf(textfile,"%d",&temp);
                        for(c1 = 0 ; c1 < 2 ; c1++){
                               ptemp = buffer + offsets[ver].convertammo + (temp * 4) + (c1 * 2);
                               fscanf(textfile,"%d",&temp2);
                               *(unsigned short*) ptemp = temp2;
                        }
                        ctemp = getc(textfile);
                        continue;
                }

                if(stricmp("Item",stemp)){
			printf("Error while parsing! (%s)\n",stemp);
			fclose(textfile);
			return(1);
                }

                fscanf(textfile,"%d",&temp);

                ctemp = getc(textfile);
                for(c1 = 0 ; ; c1 += 2){
                        ctemp = getc(textfile);
                        if(ctemp == 0x22) ctemp = getc(textfile);
                        if(ctemp == '\t') ctemp = 0;
                        if(ctemp > 0x20) ctemp++;
                        *(descbuffer + (temp * 0x0320) + c1) = ctemp;
                        if(!ctemp) break;
                }
                for(c1 = 0 ; ; c1 += 2){
                        ctemp = getc(textfile);
                        if(ctemp == 0x22) ctemp = getc(textfile);
                        if(ctemp == '\t') ctemp = 0;
                        if(ctemp > 0x20) ctemp++;
                        *(descbuffer + (temp * 0x0320) + c1 + 0xa0) = ctemp;
                        if(!ctemp) break;
                }
                for(c1 = 0 ; ; c1 += 2){
                        ctemp = getc(textfile);
                        if(ctemp == 0x22) ctemp = getc(textfile);
                        if(ctemp == '\t') ctemp = 0;
                        if(ctemp > 0x20) ctemp++;
                        *(descbuffer + (temp * 0x0320) + c1 + 0x0140) = ctemp;
                        if(!ctemp) break;
                }
                for(c1 = 0 ; ; c1 += 2){
                        ctemp = getc(textfile);
                        if(ctemp == 0x22) ctemp = getc(textfile);
                        if(ctemp == '\t') ctemp = 0;
                        if(ctemp > 0x20) ctemp++;
                        if(braybuffer) *(braybuffer + (temp * 0x0320) + c1) = ctemp;
                        if(!ctemp) break;
                }
                for(c1 = 0 ; ; c1 += 2){
                        ctemp = getc(textfile);
                        if(ctemp == 0x22) ctemp = getc(textfile);
                        if(ctemp == '\t') ctemp = 0;
                        if(ctemp > 0x20) ctemp++;
                        if(braybuffer) *(braybuffer + (temp * 0x0320) + c1 + 0xa0) = ctemp;
                        if(!ctemp) break;
                }

                for(c1 = 0 ; c1 < MAXSUBDATA ; c1++){
                        for(c2 = 0 ; ; c2++){
                               if(!attriblist[c1][c2].size) break;
                               if(temp < attribtype[c1 + ver * MAXSUBDATA].start || temp > attribtype[c1 + ver * MAXSUBDATA].end){
                                    fscanf(textfile,"%*s");
                               }
                               else{
                                    ptemp = (buffer + attribtype[c1 + ver * MAXSUBDATA].offset + ((temp - attribtype[c1 + ver * MAXSUBDATA].start) * attribtype[c1 + ver * MAXSUBDATA].size) + attriblist[c1][c2].offset);
                                    fscanf(textfile,"%d",&temp2);
                                    switch (attriblist[c1][c2].size){
                                           case 1:
                                                *(unsigned char*) ptemp = temp2;
                                                break;
                                           case -1:
                                                *(signed char*) ptemp = temp2;
                                                break;
                                           case 2:
                                                *(unsigned short*) ptemp = temp2;
                                                break;
                                           case -2:
                                                *(signed short*) ptemp = temp2;
                                                break;
                                    }
                               }
                        }
                }

                for(c1 = 0 ; c1 < MAXATTACH ; c1++){
                        if(temp > attribtype[ver * MAXSUBDATA].end){
				fscanf(textfile,"%*s");
			}
			else{
	                	if(fscanf(textfile,"%d",&temp2) == EOF) goto exit;
                                if(temp2) addattach(temp,attachtypes[c1].offset);
			}
		}
                ctemp = getc(textfile);
        }

exit:
        printf("\nSaving descriptions...");
        if(!(descfile = fopen(offsets[ver].itemname,"wb"))){
                printf("Error writing item descriptions.\n");
                return(1);
        }
        c = fwrite(descbuffer,1,DESCSIZE,descfile);
	if (c != DESCSIZE){
                printf("Error while writting item descriptions (Wrote %d of %d bytes).\n",c,DESCSIZE);
                fclose(descfile);
	        return(1);
        }
        fclose(descfile);
        if(braybuffer){
                if(!(brayfile = fopen(offsets[ver].bobbyname,"wb"))){
                        printf("Error writing Bobby Ray's item descriptions.\n");
                        return(1);
                }
                c = fwrite(braybuffer,1,DESCSIZE,brayfile);
	        if (c != DESCSIZE){
                        printf("Error while writting Bobby Ray's item descriptions (Wrote %d of %d bytes).\n",c,DESCSIZE);
                        fclose(brayfile);
	                return(1);
                }
                fclose(brayfile);
        }

        printf("\nSorting weapon attachments...");
        buffpoint = offsets[ver].weaponattach;

        for(c = 0 ; c < MAXATTACH ; c++){
                for(c1 = 0 ; c1 < 1024 ; c1++){
                        if(attachments[c][c1]){
                                 if(buffpoint < offsets[ver].otherattach) {
                                          temp = attachments[c][c1];
                                          *(unsigned long *)(buffer + buffpoint) = attachments[c][c1];
                                 }
                                 buffpoint += 4;
                        }
                }
        }
        if(buffpoint > offsets[ver].otherattach){
                printf("\n\nWARNING:  You have %d too many weapon attachments.\nThe maximum limit is %d.  The extra attachments have been ignored.\n",(buffpoint - offsets[ver].otherattach) / 4,(offsets[ver].otherattach - offsets[ver].weaponattach) / 4);
        }
        else{
                printf("\n%d of %d attachments added...",(buffpoint - offsets[ver].weaponattach) / 4,(offsets[ver].otherattach - offsets[ver].weaponattach) / 4);
        }

        while(buffpoint < offsets[ver].otherattach){
                *(unsigned long *)(buffer + buffpoint) = temp;
                buffpoint += 4;
        }

        fclose(textfile);
        if(!(exefile = fopen(argv[1],"wb"))){
		printf("Error writing %s!\n",argv[1]);
		fclose(exefile);
		return(1);
	}
        printf("\nSaving binary data...");
        fwrite(buffer,1,buffsize,exefile);
	if(ferror(exefile)){
		printf("Error writing %s!\n",argv[1]);
		fclose(exefile);
		return(1);
	}
        fclose(exefile);
        printf("\nDone!\n");
	return(0);
}

